//	TorusGamesFlickGestureRecognizer.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesFlickGestureRecognizer.h"
#import <UIKit/UIGestureRecognizerSubclass.h>


#define FLICK_RESOLUTION_MAX_TIME		0.05	//	seconds
#define	FLICK_RESOLUTION_MIN_DISTANCE	0.05	//	intrinsic units in [-0.5, +0.5] square


//	Privately-declared methods
@interface TorusGamesFlickGestureRecognizer()
- (CGPoint)touchLocation:(UITouch *)aTouch;
@end


@implementation TorusGamesFlickGestureRecognizer
{
	NSTimeInterval	itsStartTime;		//	in seconds
	NSTimeInterval	itsDuration;		//	in seconds
	CGPoint			itsStartPoint;		//	in [-0.5, +0.5] coordinates
	CGPoint			itsDisplacement;	//	in [-0.5, +0.5] coordinates
}


- (id)initWithTarget:(id)target action:(SEL)action
{
	self = [super initWithTarget:target action:action];
	if (self != nil)
	{
		itsStartTime	= 0.0;
		itsDuration		= 0.0;
		itsStartPoint	= CGPointZero;
		itsDisplacement	= CGPointZero;
	}
	return self;
}


- (NSTimeInterval)startTime	//	in seconds
{
	return itsStartTime;
}

- (NSTimeInterval)duration
{
	return itsDuration;
}

- (CGPoint)startPoint	//	in [-0.5, +0.5] coordinates
{
	return itsStartPoint;
}

- (CGPoint)displacement	//	in [-0.5, +0.5] coordinates
{
	return itsDisplacement;
}


- (void)reset
{
	[super reset];
	
	itsStartTime	= 0.0;
	itsDuration		= 0.0;
	itsStartPoint	= CGPointZero;
	itsDisplacement	= CGPointZero;
}

- (void)touchesBegan:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
//	NSSet	*theSetOfNewTouches;
	NSSet	*theSetOfAllTouches;
	UITouch	*theTouch;

	[super touchesBegan:touches withEvent:event];
	
	//	For notational clarity
//	theSetOfNewTouches = touches;
	theSetOfAllTouches = [event touchesForGestureRecognizer:self];
	
	//	Two or more fingers aren't allowed.
	if ([theSetOfAllTouches count] > 1)
		[self setState:UIGestureRecognizerStateFailed];

	//	Fetch the unique touch.
	theTouch = [touches anyObject];

	//	Note when and where the touch began.
	itsStartTime	= [theTouch timestamp];
	itsDuration		= 0.0;			//	redundant with -reset
	itsStartPoint	= [self touchLocation:theTouch];
	itsDisplacement	= CGPointZero;	//	redundant with -reset
}

- (void)touchesMoved:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	UITouch	*theTouch;
	CGPoint	theTouchPoint;
	CGFloat	theDistanceSquared;

	[super touchesMoved:touches withEvent:event];

	//	The -touchesBegan method (above) rejects multi-touch gestures,
	//	so [touches anyObject] always returns the same touch.
	theTouch = [touches anyObject];
	
	//	Where is theTouch now?
	theTouchPoint = [self touchLocation:theTouch];
	
	//	How far has the user's finger moved since the touch began?
	itsDisplacement.x = theTouchPoint.x - itsStartPoint.x;
	itsDisplacement.y = theTouchPoint.y - itsStartPoint.y;
	theDistanceSquared	= itsDisplacement.x * itsDisplacement.x
						+ itsDisplacement.y * itsDisplacement.y;
	
	//	How much time has elapsed?
	itsDuration = [theTouch timestamp] - itsStartTime;
	
	//	To be considered a flick, theTouch must move at least FLICK_RESOLUTION_MIN_DISTANCE
	//	in at most FLICK_RESOLUTION_MAX_TIME seconds.
	//
	//	Technical Note:  Avoid the temptation to set a timer.
	//	Even if you cancel the timer, messages can arrive in bad ways.
	//	For more details, see comments in KaleidoPaint's KPDrawingView.m,
	//	where it sets up its KPDoubleTapGestureRecognizer.
	//
	if (itsDuration > FLICK_RESOLUTION_MAX_TIME)
		[self setState:UIGestureRecognizerStateFailed];
	else
	if (theDistanceSquared > FLICK_RESOLUTION_MIN_DISTANCE*FLICK_RESOLUTION_MIN_DISTANCE)
		[self setState:UIGestureRecognizerStateRecognized];
	else
		;	//	Remain in UIGestureRecognizerStatePossible.
}

- (void)touchesEnded:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	[super touchesEnded:touches withEvent:event];

	[self setState:UIGestureRecognizerStateFailed];
}

- (void)touchesCancelled:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	[super touchesEnded:touches withEvent:event];

	[self setState:UIGestureRecognizerStateFailed];
}


- (CGPoint)touchLocation:(UITouch *)aTouch
{
	UIView	*theView;
	CGSize	theViewSize;
	CGPoint	theTouchPoint;
	
	//	Convert theTouchPoint from view coordinates to intrinsic [-0.5, +0.5] coordinates.
	
	//	Fetch the view and its size.
	theView		= [self view];
	theViewSize	= [theView bounds].size;	//	(w,h)

	//	touch location in view coordinates (0,0) to (w,h)
	theTouchPoint = [aTouch locationInView:theView];

	//	touch location in (-w/2, -h/2) to (+w/2, +h/2) coordinates
	theTouchPoint.x -= 0.5 * theViewSize.width;
	theTouchPoint.y -= 0.5 * theViewSize.height;

	//	touch location in (-0.5, -0.5) to (+0.5, +0.5) coordinates
	theTouchPoint.x /= theViewSize.width;
	theTouchPoint.y /= theViewSize.height;
	
	//	Flip from iOS's top-down coordinates
	//	to the Torus Games' bottom-up coordinates.
	theTouchPoint.y = - theTouchPoint.y;
	
	return theTouchPoint;
}


@end
